<?php
require_once __DIR__ . '/../config/database.php';
require_once __DIR__ . '/../config/cpanel.php';

/**
 * Kuaför Yönetim Sınıfı
 * Kuaför oluşturma, düzenleme, silme işlemleri
 */
class Kuafor {
    private $db;
    private $cpanel;
    
    public function __construct() {
        $this->db = Database::getInstance();
        $this->cpanel = new CpanelAPI();
    }
    
    /**
     * Yeni kuaför oluştur
     */
    public function create($data) {
        try {
            // Subdomain kontrolü
            $subdomain = $this->generateSubdomain($data['isletme_adi']);
            $availability = $this->cpanel->checkSubdomainAvailability($subdomain);
            
            if (!$availability['available']) {
                return [
                    'success' => false,
                    'error' => $availability['error']
                ];
            }
            
            // Veritabanına kaydet
            $sql = "INSERT INTO kuaforler (
                isletme_adi, subdomain, sahip_adi, telefon, email, 
                adres, sehir, ilce, admin_username, admin_password,
                karsilama_yazisi, alt_yazi, seo_keywords, 
                sosyal_medya, calisma_saatleri
            ) VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?)";
            
            $params = [
                $data['isletme_adi'],
                $subdomain,
                $data['sahip_adi'],
                $data['telefon'] ?? '',
                $data['email'] ?? '',
                $data['adres'] ?? '',
                $data['sehir'] ?? '',
                $data['ilce'] ?? '',
                $data['admin_username'],
                password_hash($data['admin_password'], PASSWORD_DEFAULT),
                $data['karsilama_yazisi'] ?? 'Hoş Geldiniz',
                $data['alt_yazi'] ?? '',
                $this->generateSEOKeywords($data),
                json_encode($data['sosyal_medya'] ?? []),
                json_encode($data['calisma_saatleri'] ?? $this->getDefaultWorkingHours())
            ];
            
            $stmt = $this->db->query($sql, $params);
            
            if (!$stmt) {
                return [
                    'success' => false,
                    'error' => 'Veritabanı hatası'
                ];
            }
            
            $kuafor_id = $this->db->lastInsertId();
            
            // cPanel'de subdomain oluştur
            $cpanel_result = $this->cpanel->createSubdomain($subdomain);
            
            if (!$cpanel_result['success']) {
                // Veritabanından geri al
                $this->delete($kuafor_id);
                return [
                    'success' => false,
                    'error' => 'Subdomain oluşturulamadı: ' . ($cpanel_result['error'] ?? 'Bilinmeyen hata')
                ];
            }
            
            // Kuaför klasörünü oluştur ve dosyaları kopyala
            $this->setupKuaforFiles($subdomain, $kuafor_id);
            
            // Varsayılan hizmetleri ekle
            $this->createDefaultServices($kuafor_id);
            
            // Varsayılan stilist ekle (admin)
            $this->createDefaultStylist($kuafor_id, $data);
            
            // Log kaydı
            $this->logAction($kuafor_id, 'kuafor_olusturuldu', "Yeni kuaför oluşturuldu: {$data['isletme_adi']}");
            
            return [
                'success' => true,
                'kuafor_id' => $kuafor_id,
                'subdomain' => $subdomain,
                'url' => "https://{$subdomain}.emrebaytas.com.tr",
                'admin_url' => "https://{$subdomain}.emrebaytas.com.tr/admin"
            ];
            
        } catch (Exception $e) {
            return [
                'success' => false,
                'error' => 'Sistem hatası: ' . $e->getMessage()
            ];
        }
    }
    
    /**
     * Kuaför listesi
     */
    public function getList($filters = []) {
        $sql = "SELECT * FROM kuaforler WHERE 1=1";
        $params = [];
        
        if (!empty($filters['durum'])) {
            $sql .= " AND durum = ?";
            $params[] = $filters['durum'];
        }
        
        if (!empty($filters['sehir'])) {
            $sql .= " AND sehir = ?";
            $params[] = $filters['sehir'];
        }
        
        if (!empty($filters['search'])) {
            $sql .= " AND (isletme_adi LIKE ? OR sahip_adi LIKE ?)";
            $search = '%' . $filters['search'] . '%';
            $params[] = $search;
            $params[] = $search;
        }
        
        $sql .= " ORDER BY created_at DESC";
        
        if (!empty($filters['limit'])) {
            $sql .= " LIMIT ?";
            $params[] = (int)$filters['limit'];
        }
        
        $stmt = $this->db->query($sql, $params);
        return $stmt ? $stmt->fetchAll() : [];
    }
    
    /**
     * Kuaför detayı
     */
    public function getById($id) {
        $sql = "SELECT * FROM kuaforler WHERE id = ?";
        $stmt = $this->db->query($sql, [$id]);
        return $stmt ? $stmt->fetch() : null;
    }
    
    /**
     * Subdomain ile kuaför bul
     */
    public function getBySubdomain($subdomain) {
        $sql = "SELECT * FROM kuaforler WHERE subdomain = ? AND durum = 'aktif'";
        $stmt = $this->db->query($sql, [$subdomain]);
        return $stmt ? $stmt->fetch() : null;
    }
    
    /**
     * Kuaför güncelle
     */
    public function update($id, $data) {
        try {
            $sql = "UPDATE kuaforler SET 
                isletme_adi = ?, sahip_adi = ?, telefon = ?, email = ?,
                adres = ?, sehir = ?, ilce = ?, karsilama_yazisi = ?,
                alt_yazi = ?, tema_renk = ?, sosyal_medya = ?,
                calisma_saatleri = ?, updated_at = CURRENT_TIMESTAMP
                WHERE id = ?";
            
            $params = [
                $data['isletme_adi'],
                $data['sahip_adi'],
                $data['telefon'] ?? '',
                $data['email'] ?? '',
                $data['adres'] ?? '',
                $data['sehir'] ?? '',
                $data['ilce'] ?? '',
                $data['karsilama_yazisi'] ?? 'Hoş Geldiniz',
                $data['alt_yazi'] ?? '',
                $data['tema_renk'] ?? 'mavi',
                json_encode($data['sosyal_medya'] ?? []),
                json_encode($data['calisma_saatleri'] ?? []),
                $id
            ];
            
            $stmt = $this->db->query($sql, $params);
            
            if ($stmt) {
                $this->logAction($id, 'kuafor_guncellendi', 'Kuaför bilgileri güncellendi');
                return ['success' => true];
            }
            
            return ['success' => false, 'error' => 'Güncelleme başarısız'];
            
        } catch (Exception $e) {
            return ['success' => false, 'error' => $e->getMessage()];
        }
    }
    
    /**
     * Kuaför sil
     */
    public function delete($id) {
        try {
            $kuafor = $this->getById($id);
            if (!$kuafor) {
                return ['success' => false, 'error' => 'Kuaför bulunamadı'];
            }
            
            // Subdomain'i sil
            $this->cpanel->deleteSubdomain($kuafor['subdomain']);
            
            // Veritabanından sil (CASCADE ile ilişkili veriler de silinir)
            $sql = "DELETE FROM kuaforler WHERE id = ?";
            $stmt = $this->db->query($sql, [$id]);
            
            if ($stmt) {
                $this->logAction(null, 'kuafor_silindi', "Kuaför silindi: {$kuafor['isletme_adi']}");
                return ['success' => true];
            }
            
            return ['success' => false, 'error' => 'Silme işlemi başarısız'];
            
        } catch (Exception $e) {
            return ['success' => false, 'error' => $e->getMessage()];
        }
    }
    
    /**
     * Subdomain oluştur
     */
    private function generateSubdomain($isletme_adi) {
        // Türkçe karakterleri değiştir
        $subdomain = $this->turkishToEnglish($isletme_adi);
        
        // Sadece harf, rakam ve tire bırak
        $subdomain = preg_replace('/[^a-z0-9-]/', '', strtolower($subdomain));
        
        // Çoklu tireleri tek tire yap
        $subdomain = preg_replace('/-+/', '-', $subdomain);
        
        // Başında ve sonunda tire varsa sil
        $subdomain = trim($subdomain, '-');
        
        // Minimum 3, maksimum 20 karakter
        if (strlen($subdomain) < 3) {
            $subdomain .= rand(100, 999);
        }
        
        if (strlen($subdomain) > 20) {
            $subdomain = substr($subdomain, 0, 20);
        }
        
        // Benzersizlik kontrolü
        $original = $subdomain;
        $counter = 1;
        
        while ($this->subdomainExists($subdomain)) {
            $subdomain = $original . $counter;
            $counter++;
        }
        
        return $subdomain;
    }
    
    /**
     * Türkçe karakterleri İngilizce'ye çevir
     */
    private function turkishToEnglish($text) {
        $turkish = ['ç', 'ğ', 'ı', 'ö', 'ş', 'ü', 'Ç', 'Ğ', 'I', 'İ', 'Ö', 'Ş', 'Ü'];
        $english = ['c', 'g', 'i', 'o', 's', 'u', 'c', 'g', 'i', 'i', 'o', 's', 'u'];
        return str_replace($turkish, $english, $text);
    }
    
    /**
     * Subdomain var mı kontrol et
     */
    private function subdomainExists($subdomain) {
        $sql = "SELECT COUNT(*) FROM kuaforler WHERE subdomain = ?";
        $stmt = $this->db->query($sql, [$subdomain]);
        return $stmt && $stmt->fetchColumn() > 0;
    }
    
    /**
     * SEO anahtar kelimeleri oluştur
     */
    private function generateSEOKeywords($data) {
        $keywords = [];
        
        // Şehir + kuaför kombinasyonları
        if (!empty($data['sehir'])) {
            $keywords[] = strtolower($data['sehir']) . ' kuaför';
            $keywords[] = strtolower($data['sehir']) . ' berber';
            $keywords[] = strtolower($data['sehir']) . ' kuaför salonu';
        }
        
        // İlçe + kuaför kombinasyonları
        if (!empty($data['ilce'])) {
            $keywords[] = strtolower($data['ilce']) . ' kuaför';
            $keywords[] = strtolower($data['ilce']) . ' berber';
        }
        
        // Genel kelimeler
        $keywords[] = 'saç kesimi';
        $keywords[] = 'sakal tıraşı';
        $keywords[] = 'kuaför randevu';
        $keywords[] = 'berber randevu';
        $keywords[] = 'online randevu';
        
        return implode(', ', array_unique($keywords));
    }
    
    /**
     * Varsayılan çalışma saatleri
     */
    private function getDefaultWorkingHours() {
        return [
            'pazartesi' => ['aktif' => true, 'acilis' => '09:00', 'kapanis' => '18:00'],
            'sali' => ['aktif' => true, 'acilis' => '09:00', 'kapanis' => '18:00'],
            'carsamba' => ['aktif' => true, 'acilis' => '09:00', 'kapanis' => '18:00'],
            'persembe' => ['aktif' => true, 'acilis' => '09:00', 'kapanis' => '18:00'],
            'cuma' => ['aktif' => true, 'acilis' => '09:00', 'kapanis' => '18:00'],
            'cumartesi' => ['aktif' => true, 'acilis' => '09:00', 'kapanis' => '17:00'],
            'pazar' => ['aktif' => false, 'acilis' => '', 'kapanis' => '']
        ];
    }
    
    /**
     * Kuaför dosyalarını kur
     */
    private function setupKuaforFiles($subdomain, $kuafor_id) {
        // Bu fonksiyon kuaför klasörünü oluşturup template dosyalarını kopyalar
        // Şimdilik boş bırakıyoruz, sonra implement edeceğiz
    }
    
    /**
     * Varsayılan hizmetleri oluştur
     */
    private function createDefaultServices($kuafor_id) {
        $services = [
            ['Saç Kesimi', 'Profesyonel saç kesimi hizmeti', 30, 100.00],
            ['Sakal Tıraşı', 'Geleneksel sakal tıraşı', 15, 50.00],
            ['Saç Yıkama', 'Şampuan ve saç bakımı', 10, 30.00],
            ['Fön Çekimi', 'Profesyonel fön ve şekillendirme', 20, 80.00]
        ];
        
        foreach ($services as $index => $service) {
            $sql = "INSERT INTO hizmetler (kuafor_id, hizmet_adi, aciklama, sure_dakika, fiyat, sira_no) 
                    VALUES (?, ?, ?, ?, ?, ?)";
            $this->db->query($sql, [$kuafor_id, $service[0], $service[1], $service[2], $service[3], $index + 1]);
        }
    }
    
    /**
     * Varsayılan stilist oluştur
     */
    private function createDefaultStylist($kuafor_id, $data) {
        $sql = "INSERT INTO stilistler (kuafor_id, isim_soyisim, uzmanlik_alani, username, password) 
                VALUES (?, ?, ?, ?, ?)";
        
        $params = [
            $kuafor_id,
            $data['sahip_adi'],
            'Genel',
            $data['admin_username'],
            password_hash($data['admin_password'], PASSWORD_DEFAULT)
        ];
        
        $this->db->query($sql, $params);
    }
    
    /**
     * İşlem logla
     */
    private function logAction($kuafor_id, $islem_tipi, $aciklama) {
        $sql = "INSERT INTO sistem_loglari (kuafor_id, islem_tipi, aciklama, ip_adresi, user_agent) 
                VALUES (?, ?, ?, ?, ?)";
        
        $params = [
            $kuafor_id,
            $islem_tipi,
            $aciklama,
            $_SERVER['REMOTE_ADDR'] ?? '',
            $_SERVER['HTTP_USER_AGENT'] ?? ''
        ];
        
        $this->db->query($sql, $params);
    }
    
    /**
     * Dashboard istatistikleri
     */
    public function getDashboardStats() {
        $stats = [];
        
        // Toplam kuaför sayısı
        $sql = "SELECT COUNT(*) as toplam FROM kuaforler";
        $stmt = $this->db->query($sql);
        $stats['toplam_kuafor'] = $stmt ? $stmt->fetchColumn() : 0;
        
        // Aktif kuaför sayısı
        $sql = "SELECT COUNT(*) as aktif FROM kuaforler WHERE durum = 'aktif'";
        $stmt = $this->db->query($sql);
        $stats['aktif_kuafor'] = $stmt ? $stmt->fetchColumn() : 0;
        
        // Bu ayki yeni kuaförler
        $sql = "SELECT COUNT(*) as yeni FROM kuaforler WHERE DATE(created_at) >= DATE_SUB(CURDATE(), INTERVAL 30 DAY)";
        $stmt = $this->db->query($sql);
        $stats['aylik_yeni'] = $stmt ? $stmt->fetchColumn() : 0;
        
        // Toplam randevu sayısı
        $sql = "SELECT COUNT(*) as toplam FROM randevular";
        $stmt = $this->db->query($sql);
        $stats['toplam_randevu'] = $stmt ? $stmt->fetchColumn() : 0;
        
        return $stats;
    }
}
?>
