<?php
/**
 * Kuaför Sistemi - Gelişmiş Veritabanı Kurulum Scripti
 * Bu dosya veritabanı tablolarını oluşturur ve varsayılan verileri ekler
 * MySQL 5.7+ uyumlu, güvenli index oluşturma ile
 */

require_once __DIR__ . '/../config/database.php';

// Hata raporlamayı aç
error_reporting(E_ALL);
ini_set('display_errors', 1);

class DatabaseInstaller {
    private $connection;
    private $successCount = 0;
    private $errorCount = 0;
    private $warningCount = 0;
    
    public function __construct() {
        $db = Database::getInstance();
        $this->connection = $db->getConnection();
    }
    
    /**
     * Ana kurulum fonksiyonu
     */
    public function install() {
        $this->printHeader();
        
        try {
            $this->testConnection();
            $this->createTables();
            $this->createIndexes();
            $this->insertDefaultData();
            $this->createTestData();
            $this->showResults();
            $this->printFooter();
            
        } catch (Exception $e) {
            $this->printError("Kritik Hata: " . $e->getMessage());
            $this->printError("Stack Trace: " . $e->getTraceAsString());
        }
    }
    
    /**
     * Veritabanı bağlantısını test et
     */
    private function testConnection() {
        $this->printInfo("Veritabanı bağlantısı test ediliyor...");
        
        // Bağlantı testi
        $result = $this->connection->query("SELECT 1");
        if (!$result) {
            throw new Exception("Veritabanı bağlantısı başarısız!");
        }
        
        // Veritabanı adını al
        $dbName = $this->connection->query("SELECT DATABASE()")->fetchColumn();
        
        $this->printSuccess("✅ Veritabanı bağlantısı başarılı!");
        $this->printInfo("Veritabanı: $dbName");
        
        // MySQL versiyonunu kontrol et
        $version = $this->connection->query("SELECT VERSION()")->fetchColumn();
        $this->printInfo("MySQL Versiyonu: $version");
    }
    
    /**
     * Tabloları oluştur
     */
    private function createTables() {
        $this->printInfo("SQL dosyası okunuyor...");
        
        $sqlFile = __DIR__ . '/schema.sql';
        if (!file_exists($sqlFile)) {
            throw new Exception("SQL dosyası bulunamadı: $sqlFile");
        }
        
        $sql = file_get_contents($sqlFile);
        if ($sql === false) {
            throw new Exception("SQL dosyası okunamadı");
        }
        
        // SQL komutlarını ayır ve temizle
        $statements = $this->parseSQLStatements($sql);
        
        $this->printInfo("Toplam " . count($statements) . " SQL komutu bulundu");
        $this->printInfo("Tablolar oluşturuluyor...");
        
        foreach ($statements as $index => $statement) {
            $this->executeStatement($statement, $index + 1);
        }
    }
    
    /**
     * Index'leri güvenli şekilde oluştur
     */
    private function createIndexes() {
        $this->printInfo("Performans index'leri oluşturuluyor...");
        
        // Önce tabloların var olup olmadığını kontrol et
        $tables = ['randevular', 'hizmetler', 'stilistler', 'destek_talepleri', 'iletisim_formlari'];
        $existingTables = [];
        
        foreach ($tables as $table) {
            try {
                $result = $this->connection->query("SHOW TABLES LIKE '$table'");
                if ($result && $result->rowCount() > 0) {
                    $existingTables[] = $table;
                }
            } catch (Exception $e) {
                // Tablo yok, devam et
            }
        }
        
        if (empty($existingTables)) {
            $this->printWarning("⚠️ Tablolar henüz oluşturulmamış, index'ler atlanıyor");
            return;
        }
        
        $indexes = [];
        
        if (in_array('randevular', $existingTables)) {
            $indexes['idx_randevular_tarih'] = 'CREATE INDEX idx_randevular_tarih ON randevular(randevu_tarihi)';
            $indexes['idx_randevular_stilist'] = 'CREATE INDEX idx_randevular_stilist ON randevular(stilist_id)';
            $indexes['idx_randevular_durum'] = 'CREATE INDEX idx_randevular_durum ON randevular(durum)';
        }
        
        if (in_array('hizmetler', $existingTables)) {
            $indexes['idx_hizmetler_kuafor'] = 'CREATE INDEX idx_hizmetler_kuafor ON hizmetler(kuafor_id)';
        }
        
        if (in_array('stilistler', $existingTables)) {
            $indexes['idx_stilistler_kuafor'] = 'CREATE INDEX idx_stilistler_kuafor ON stilistler(kuafor_id)';
        }
        
        if (in_array('destek_talepleri', $existingTables)) {
            $indexes['idx_destek_durum'] = 'CREATE INDEX idx_destek_durum ON destek_talepleri(durum)';
        }
        
        if (in_array('iletisim_formlari', $existingTables)) {
            $indexes['idx_iletisim_durum'] = 'CREATE INDEX idx_iletisim_durum ON iletisim_formlari(durum)';
        }
        
        if (empty($indexes)) {
            $this->printWarning("⚠️ Index oluşturulacak tablo bulunamadı");
            return;
        }
        
        foreach ($indexes as $indexName => $sql) {
            $this->createIndexSafely($indexName, $sql);
        }
    }
    
    /**
     * Index'i güvenli şekilde oluştur
     */
    private function createIndexSafely($indexName, $sql) {
        try {
            // Önce index'in var olup olmadığını kontrol et
            $checkSql = "SHOW INDEX FROM " . $this->getTableFromIndexSQL($sql) . " WHERE Key_name = ?";
            $stmt = $this->connection->prepare($checkSql);
            $stmt->execute([$indexName]);
            
            if ($stmt->rowCount() > 0) {
                $this->printWarning("ℹ️ Index zaten mevcut: $indexName");
                $this->warningCount++;
                return;
            }
            
            // Index'i oluştur
            $this->connection->exec($sql);
            $this->printSuccess("✅ Index oluşturuldu: $indexName");
            $this->successCount++;
            
        } catch (PDOException $e) {
            if (strpos($e->getMessage(), 'Duplicate key name') !== false) {
                $this->printWarning("ℹ️ Index zaten mevcut: $indexName");
                $this->warningCount++;
            } else {
                $this->printError("❌ Index oluşturulamadı ($indexName): " . $e->getMessage());
                $this->errorCount++;
            }
        }
    }
    
    /**
     * SQL'den tablo adını çıkar
     */
    private function getTableFromIndexSQL($sql) {
        if (preg_match('/ON\s+(\w+)\s*\(/i', $sql, $matches)) {
            return $matches[1];
        }
        return 'unknown_table';
    }
    
    /**
     * Varsayılan verileri ekle
     */
    private function insertDefaultData() {
        $this->printInfo("Varsayılan veriler kontrol ediliyor...");
        
        // Ana site ayarları var mı kontrol et
        $count = $this->connection->query("SELECT COUNT(*) FROM ana_site_ayarlari")->fetchColumn();
        
        if ($count == 0) {
            $this->printInfo("Varsayılan veriler ekleniyor...");
            // Varsayılan veriler schema.sql'de INSERT komutu ile zaten ekleniyor
            $this->printSuccess("✅ Varsayılan veriler eklendi");
        } else {
            $this->printWarning("ℹ️ Varsayılan veriler zaten mevcut ($count kayıt)");
        }
    }
    
    /**
     * Test verisi oluştur
     */
    private function createTestData() {
        $this->printInfo("Test verisi kontrol ediliyor...");
        
        // Demo kuaför var mı kontrol et
        $existingKuafor = $this->connection->query("SELECT id FROM kuaforler WHERE subdomain = 'demo'")->fetch();
        
        if (!$existingKuafor) {
            $this->printInfo("Demo kuaför oluşturuluyor...");
            $this->createDemoKuafor();
        } else {
            $this->printWarning("ℹ️ Demo kuaför zaten mevcut (ID: " . $existingKuafor['id'] . ")");
        }
    }
    
    /**
     * Demo kuaför oluştur
     */
    private function createDemoKuafor() {
        try {
            $testKuafor = [
                'isletme_adi' => 'Demo Kuaför',
                'subdomain' => 'demo',
                'sahip_adi' => 'Demo Kullanıcı',
                'telefon' => '0555 123 45 67',
                'email' => 'demo@kuafor.com',
                'adres' => 'Demo Mahallesi, Demo Sokak No:1',
                'sehir' => 'İstanbul',
                'ilce' => 'Kadıköy',
                'admin_username' => 'admin',
                'admin_password' => password_hash('123456', PASSWORD_DEFAULT),
                'karsilama_yazisi' => 'Demo Kuaföre Hoş Geldiniz',
                'alt_yazi' => 'Profesyonel kuaför hizmetleri',
                'seo_keywords' => 'istanbul kuaför, kadıköy berber, saç kesimi',
                'sosyal_medya' => json_encode([
                    'instagram' => 'https://instagram.com/demokuafor',
                    'facebook' => 'https://facebook.com/demokuafor'
                ]),
                'calisma_saatleri' => json_encode([
                    'pazartesi' => ['aktif' => true, 'acilis' => '09:00', 'kapanis' => '18:00'],
                    'sali' => ['aktif' => true, 'acilis' => '09:00', 'kapanis' => '18:00'],
                    'carsamba' => ['aktif' => true, 'acilis' => '09:00', 'kapanis' => '18:00'],
                    'persembe' => ['aktif' => true, 'acilis' => '09:00', 'kapanis' => '18:00'],
                    'cuma' => ['aktif' => true, 'acilis' => '09:00', 'kapanis' => '18:00'],
                    'cumartesi' => ['aktif' => true, 'acilis' => '09:00', 'kapanis' => '17:00'],
                    'pazar' => ['aktif' => false, 'acilis' => '', 'kapanis' => '']
                ])
            ];
            
            $sql = "INSERT INTO kuaforler (
                isletme_adi, subdomain, sahip_adi, telefon, email, adres, sehir, ilce,
                admin_username, admin_password, karsilama_yazisi, alt_yazi, seo_keywords,
                sosyal_medya, calisma_saatleri
            ) VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?)";
            
            $stmt = $this->connection->prepare($sql);
            $result = $stmt->execute([
                $testKuafor['isletme_adi'], $testKuafor['subdomain'], $testKuafor['sahip_adi'],
                $testKuafor['telefon'], $testKuafor['email'], $testKuafor['adres'],
                $testKuafor['sehir'], $testKuafor['ilce'], $testKuafor['admin_username'],
                $testKuafor['admin_password'], $testKuafor['karsilama_yazisi'], $testKuafor['alt_yazi'],
                $testKuafor['seo_keywords'], $testKuafor['sosyal_medya'], $testKuafor['calisma_saatleri']
            ]);
            
            if ($result) {
                $kuafor_id = $this->connection->lastInsertId();
                $this->printSuccess("✅ Demo kuaför oluşturuldu (ID: $kuafor_id)");
                
                $this->createDemoStylist($kuafor_id);
                $this->createDemoServices($kuafor_id);
                
            } else {
                $this->printError("❌ Demo kuaför oluşturulamadı");
            }
            
        } catch (Exception $e) {
            $this->printError("❌ Demo kuaför oluşturma hatası: " . $e->getMessage());
        }
    }
    
    /**
     * Demo stilist oluştur
     */
    private function createDemoStylist($kuafor_id) {
        try {
            $stmt = $this->connection->prepare("INSERT INTO stilistler (kuafor_id, isim_soyisim, uzmanlik_alani, username, password) VALUES (?, ?, ?, ?, ?)");
            $stmt->execute([$kuafor_id, 'Demo Stilist', 'Saç & Sakal', 'demo', password_hash('123456', PASSWORD_DEFAULT)]);
            $this->printSuccess("✅ Demo stilist oluşturuldu");
        } catch (Exception $e) {
            $this->printError("❌ Demo stilist oluşturma hatası: " . $e->getMessage());
        }
    }
    
    /**
     * Demo hizmetler oluştur
     */
    private function createDemoServices($kuafor_id) {
        try {
            $services = [
                ['Saç Kesimi', 'Profesyonel saç kesimi', 30, 100.00, 1],
                ['Sakal Tıraşı', 'Geleneksel sakal tıraşı', 15, 50.00, 2],
                ['Saç Yıkama', 'Şampuan ve bakım', 10, 30.00, 3],
                ['Fön Çekimi', 'Profesyonel fön', 20, 80.00, 4]
            ];
            
            $stmt = $this->connection->prepare("INSERT INTO hizmetler (kuafor_id, hizmet_adi, aciklama, sure_dakika, fiyat, sira_no) VALUES (?, ?, ?, ?, ?, ?)");
            
            foreach ($services as $service) {
                $stmt->execute([$kuafor_id, $service[0], $service[1], $service[2], $service[3], $service[4]]);
            }
            
            $this->printSuccess("✅ Demo hizmetler oluşturuldu (" . count($services) . " adet)");
        } catch (Exception $e) {
            $this->printError("❌ Demo hizmetler oluşturma hatası: " . $e->getMessage());
        }
    }
    
    /**
     * Sonuçları göster
     */
    private function showResults() {
        $this->printInfo("Kurulum tamamlandı!");
        $this->printSuccess("✅ Başarılı: " . $this->successCount . " işlem");
        
        if ($this->warningCount > 0) {
            $this->printWarning("⚠️ Uyarı: " . $this->warningCount . " işlem");
        }
        
        if ($this->errorCount > 0) {
            $this->printError("❌ Hatalı: " . $this->errorCount . " işlem");
        }
        
        // Tabloları listele
        echo "<h2>📊 Oluşturulan Tablolar</h2>";
        $tables = $this->connection->query("SHOW TABLES")->fetchAll(PDO::FETCH_COLUMN);
        
        foreach ($tables as $table) {
            $count = $this->connection->query("SELECT COUNT(*) FROM `$table`")->fetchColumn();
            $this->printInfo("📋 $table: $count kayıt");
        }
    }
    
    /**
     * SQL komutlarını ayrıştır
     */
    private function parseSQLStatements($sql) {
        $statements = array_filter(
            array_map('trim', explode(';', $sql)),
            function($stmt) {
                $stmt = trim($stmt);
                return !empty($stmt) && 
                       !preg_match('/^--/', $stmt) && 
                       !preg_match('/^\s*$/', $stmt);
            }
        );
        
        return $statements;
    }
    
    /**
     * SQL komutunu çalıştır
     */
    private function executeStatement($statement, $index) {
        try {
            $stmt = $this->connection->prepare($statement);
            $result = $stmt->execute();
            
            if ($result) {
                $this->successCount++;
                
                // Tablo oluşturma komutlarını göster
                if (preg_match('/CREATE TABLE.*?`?(\w+)`?/i', $statement, $matches)) {
                    $this->printSuccess("✅ Tablo oluşturuldu: {$matches[1]}");
                }
                // Insert komutlarını göster
                elseif (preg_match('/INSERT INTO.*?(\w+)/i', $statement, $matches)) {
                    $affectedRows = $stmt->rowCount();
                    $this->printSuccess("✅ Veri eklendi: {$matches[1]} ($affectedRows kayıt)");
                }
            }
        } catch (PDOException $e) {
            $this->errorCount++;
            
            // Tablo zaten varsa uyarı olarak göster
            if (strpos($e->getMessage(), 'already exists') !== false) {
                if (preg_match('/CREATE TABLE.*?`?(\w+)`?/i', $statement, $matches)) {
                    $this->printWarning("ℹ️ Tablo zaten mevcut: {$matches[1]}");
                    $this->warningCount++;
                    $this->errorCount--; // Hata sayısından çıkar
                }
            } 
            // Duplicate entry hatası
            elseif (strpos($e->getMessage(), 'Duplicate entry') !== false) {
                if (preg_match('/INSERT INTO.*?(\w+)/i', $statement, $matches)) {
                    $this->printWarning("ℹ️ Veri zaten mevcut: {$matches[1]}");
                    $this->warningCount++;
                    $this->errorCount--; // Hata sayısından çıkar
                }
            }
            else {
                $this->printError("❌ Hata (Komut $index): " . $e->getMessage());
                echo "<pre style='background:#fff0f0;padding:10px;border:1px solid red;margin:5px 0;font-size:12px;'>" . 
                     htmlspecialchars(substr($statement, 0, 200)) . "...</pre>";
            }
        }
    }
    
    /**
     * HTML çıktı fonksiyonları
     */
    private function printHeader() {
        echo "<!DOCTYPE html>
<html lang='tr'>
<head>
    <meta charset='UTF-8'>
    <meta name='viewport' content='width=device-width, initial-scale=1.0'>
    <title>Kuaför Sistemi - Gelişmiş Veritabanı Kurulumu</title>
    <style>
        body { font-family: Arial, sans-serif; max-width: 900px; margin: 50px auto; padding: 20px; background: #f8f9fa; }
        .container { background: white; padding: 30px; border-radius: 10px; box-shadow: 0 2px 10px rgba(0,0,0,0.1); }
        .success { color: #155724; background: #d4edda; padding: 10px; border: 1px solid #c3e6cb; margin: 5px 0; border-radius: 5px; }
        .error { color: #721c24; background: #f8d7da; padding: 10px; border: 1px solid #f5c6cb; margin: 5px 0; border-radius: 5px; }
        .warning { color: #856404; background: #fff3cd; padding: 10px; border: 1px solid #ffeaa7; margin: 5px 0; border-radius: 5px; }
        .info { color: #0c5460; background: #d1ecf1; padding: 10px; border: 1px solid #bee5eb; margin: 5px 0; border-radius: 5px; }
        pre { background: #f8f9fa; padding: 10px; border: 1px solid #dee2e6; overflow-x: auto; border-radius: 5px; font-size: 12px; }
        h1 { color: #343a40; text-align: center; margin-bottom: 30px; }
        h2 { color: #495057; border-bottom: 2px solid #dee2e6; padding-bottom: 10px; }
        .footer { text-align: center; margin-top: 30px; padding: 20px; background: #e9ecef; border-radius: 5px; }
        .footer a { color: #007bff; text-decoration: none; }
        .footer a:hover { text-decoration: underline; }
    </style>
</head>
<body>
    <div class='container'>
        <h1>🚀 Kuaför Sistemi - Gelişmiş Veritabanı Kurulumu</h1>";
    }
    
    private function printFooter() {
        echo "<h2>🎉 Kurulum Başarıyla Tamamlandı!</h2>";
        echo "<div class='success'>
            <h3>🔗 Erişim Linkleri:</h3>
            <p><strong>Ana Site:</strong> <a href='../index.php' target='_blank'>index.php</a></p>
            <p><strong>Demo Sayfası:</strong> <a href='../demo.html' target='_blank'>demo.html</a></p>
            <p><strong>Super Admin:</strong> <a href='../xpanel' target='_blank'>xpanel</a> (Henüz oluşturulmadı)</p>
            <p><strong>Demo Kuaför:</strong> <a href='https://demo.emrebaytas.com.tr' target='_blank'>demo.emrebaytas.com.tr</a></p>
            <p><strong>Demo Admin Giriş:</strong> Kullanıcı: admin, Şifre: 123456</p>
        </div>";
        
        echo "<div class='footer'>
            <p><strong>Kuaför Sistemi</strong> başarıyla kuruldu!</p>
            <p>Geliştirici: <a href='https://emrebaytas.com.tr' target='_blank'>Emre Baytaş</a></p>
            <p>Sistem hosting'e yüklenmeye hazır.</p>
        </div>";
        
        echo "</div></body></html>";
    }
    
    private function printSuccess($message) {
        echo "<div class='success'>$message</div>";
    }
    
    private function printError($message) {
        echo "<div class='error'>$message</div>";
    }
    
    private function printWarning($message) {
        echo "<div class='warning'>$message</div>";
    }
    
    private function printInfo($message) {
        echo "<div class='info'>$message</div>";
    }
}

// Kurulumu başlat
$installer = new DatabaseInstaller();
$installer->install();
?>
